% Simulating Transducer Field Patterns Example
%
% This example demonstrates the use of k-Wave to compute the field pattern
% generated by a curved single element transducer in two dimensions.
%
% author: Bradley Treeby
% date: 10th December 2009
% last update: 19th January 2010
%  
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010 Bradley Treeby and Ben Cox

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>. 

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% create the computational grid
Nx = 128;           % number of pixels in the x (column) direction
Nz = 128;           % number of pixels in the z (row) direction
dx = 50e-3/Nx;    	% pixel width [m]
dz = dx;     	% pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% define the properties of the propagation medium
medium.sound_speed = 1500;  % [m/s]
medium.density = 1000;      % [kg/m^3]
medium.alpha_power = 1.5;   % [dB/(MHz^y cm)]
medium.alpha_coeff = 0.75;  % [dB/(MHz^y cm)]

% create the time array
[kgrid.t_array dt] = makeTime(kgrid, medium.sound_speed);

% define a curved transducer element
source.p_mask = makeCircle(Nx, Nz, 50, 50, 30, pi/2);

% define a time varying sinusoidal source
source_freq = 0.25e6;
source_mag = 0.5;
source.p = source_mag*sin(2*pi*source_freq*kgrid.t_array);

% smooth the source
source.p = filterTimeSeries(kgrid, medium, source.p);

% define a single sensor element
sensor.mask = zeros(Nz, Nx);
sensor.mask(Nz/2, Nx/2) = 1;

% create a display mask to display the transducer
display_mask = source.p_mask;

% run the simulation
[sensor_data p_final] = kspaceFirstOrder2D(kgrid, medium, source, sensor, 'DisplayMask', display_mask);

% =========================================================================
% VISUALISATION
% =========================================================================

% add the source mask onto the recorded wave-field
p_final(source.p_mask ~= 0) = 1;

% plot the final wave-field
figure;
imagesc(kgrid.x(1,:)*1e3, kgrid.z(:,1)*1e3, p_final, [-1 1]);
colormap(getColorMap);
ylabel('z-position [mm]');
xlabel('x-position [mm]');
axis image;